/**
 * \file: init_daemon_state_cmd.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: authorization level daemon
 *
 * \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
 *
 * \copyright (c) 2017 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/
#include "init_daemon_state_cmd/init_daemon_state_cmd.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <libgen.h>
#include <linux/limits.h>

#include "util/helper.h"
#include "util/cmdline_parser.h"
#include "util/cfgfile_parser.h"

#include "control/configuration_defaults.h"
#include "model/daemon_model.h"


static cfg_item_t state_val_cfg		= {.spec = &state_val_cfg_spec, .cmdline = &init_level_cmdline_spec };
static cfg_item_t state_path_cfg 	= {.spec = &state_path_cfg_spec, .cmdline = &state_path_cmdline_spec };
static cfg_item_t conffile_cfg 		= {.spec = &conf_file_cfg_spec, .cmdline = &conffile_cmdline_spec };
static cfg_item_t help_cfg 		= {.spec = &help_cfg_spec, .cmdline = &help_cmdline_spec };

static cfg_item_t *const init_state_cfgs[] = {
		&state_val_cfg,
		&state_path_cfg,
		&conffile_cfg,
		&help_cfg
};

static error_code_t init_daemon_state_cmd_init(void);
static error_code_t init_daemon_state_cmd_parse_args(const char *binary_name, int argc, char *argv[]);
static error_code_t init_daemon_state_cmd_start(void);
static void init_daemon_state_cmd_print_help(void);
static void init_daemon_state_cmd_print_usage(void);
static void init_daemon_state_cmd_deinit(void);

static error_code_t init_daemon_state_cmd_init(void)
{
	logger_init_console_only(LOGGER_LEVEL_ERROR);
	return RESULT_OK;
}

static error_code_t init_daemon_state_cmd_parse_args(const char *binary_name, int argc, char *argv[])
{
	error_code_t result = RESULT_OK;
	int init_state_cfgs_cnt = ARRAY_SIZE(init_state_cfgs);

	(void) binary_name;

	helper_items_init(init_state_cfgs, init_state_cfgs_cnt);

	result = cmdline_parser_parse_cmd_line(init_state_cfgs, init_state_cfgs_cnt, argv, argc);

	if (result == RESULT_INVALID_ARGS)
	{
		init_daemon_state_cmd_print_usage();
		return RESULT_INVALID_ARGS;
	}

	if (helper_get_flag(&help_cfg))
	{
		init_daemon_state_cmd_print_help();
		return RESULT_HELP_PRINTED;
	}

	if (cfgfile_parser_is_any_cfg_item_unset(init_state_cfgs, init_state_cfgs_cnt))
		result = cfgfile_parser_parse_config_file(helper_get_str_value(&conffile_cfg),
				init_state_cfgs, init_state_cfgs_cnt);

	return result;
}

static void init_daemon_state_cmd_print_help(void)
{
	printf("\n");
	printf("Authorization Level Daemon Initialization Utility - Used to create configurations and\n"
			"signatures for the ADIT authorization level daemon.\n\n");
	printf("Usage: ald_init %s [OPTIONS...]\n",init_daemon_state_cmd_vtable.command);
	printf("This command creates the initial daemon state file for the authorization level daemon.\n\n");
	printf("\t"INIT_LEVEL_CMD_SHORT", "INIT_LEVEL_CMD_LONG"\t\tThe initial security level the system is delivered with\n"
			"\t\t\t\tand it needs to be a positive integer value.\n");
	printf("\t"STATE_PATH_CMD_SHORT", "STATE_PATH_CMD_LONG"\t\tFilename and path where the state is stored in\n");
	printf("\t"CONFFILE_CMD_SHORT", "CONFFILE_CMD_LONG"\t\tLoad the state file path & filename from this config file\n");
	printf("\t"HELP_CMD_SHORT", "HELP_CMD_LONG"\t\tDisplays this help and exits.\n\n");
	printf("If no intial security level is passed, the default level is considered as '%d'.\n"
			"If no filename & path for state file is passed, the path is read from config file passed.\n"
			"If no config file is passed default config file (%s) is used.\n"
			"If filename & path for state file is not available in config file, then the state file\n"
			"is stored in the default location (%s).\n\n", helper_get_U32_default(state_val_cfg.spec),
			helper_get_str_default(conffile_cfg.spec),helper_get_str_default(state_path_cfg.spec));
}

static void init_daemon_state_cmd_print_usage(void)
{
	printf("\n");
	printf("Usage: ald_init %s [OPTIONS...]\n\n",init_daemon_state_cmd_vtable.command);
	printf("\t"INIT_LEVEL_CMD_SHORT", "INIT_LEVEL_CMD_LONG"\t\tThe initial security level the system is delivered with\n"
			"\t\t\t\tand it needs to be a positive integer value.\n");
	printf("\t"STATE_PATH_CMD_SHORT", "STATE_PATH_CMD_LONG"\t\tFilename and path where the state is stored in\n");
	printf("\t"CONFFILE_CMD_SHORT", "CONFFILE_CMD_LONG"\t\tLoad the state file path & filename from this config file\n");
	printf("\t"HELP_CMD_SHORT", "HELP_CMD_LONG"\t\tDisplays this help and exits.\n\n");
}

static error_code_t init_daemon_state_cmd_start(void)
{
	struct stat st={0};
	char path[PATH_MAX]={0};
	char *dir=NULL;
	int fd=0;
	security_level_t level = helper_get_U32_value(&state_val_cfg);
	const char *state_path = helper_get_str_value(&state_path_cfg);

	persistent_model_t daemon_persistent_model={
			.current_persisted_level=level,
			.targeted_persisted_level=level
	};

	if (level > MAX_LEVEL) {
		printf("ERROR: level %d is exceeding max allowed value.\n",level);
		return RESULT_INVALID_LEVEL;
	}

	strncpy(path, state_path,PATH_MAX);
	dir=dirname(path);
	if (stat(dir, &st) == -1)
	{
		if (ENOENT==errno)
		{
			mkdir(dir, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
		}
		else
		{
			printf("Unable to stat output file %s. Message: %s\n", state_path, strerror(errno));
			return RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES;
		}
	}

	fd=open(state_path,O_CREAT|O_WRONLY,00666);
	if (fd==-1)
	{
		printf("Unable to create output file %s. Message: %s\n", state_path, strerror(errno));
		return RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES;
	}

	if (write(fd, &daemon_persistent_model,sizeof(persistent_model_t))!= (int)sizeof(persistent_model_t))
	{
		printf("Unable to store the initial daemon state into file %s. Message: %s\n", state_path, strerror(errno));
		close(fd);
		return RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES;
	}

	close(fd);
	printf("Created the initial daemon state and stored it under %s. The initial security level is set to %d.\n",
			state_path, (int)level);
	return RESULT_OK;
}

static void init_daemon_state_cmd_deinit(void)
{
	helper_items_free(init_state_cfgs, ARRAY_SIZE(init_state_cfgs));
}

command_vtable_t init_daemon_state_cmd_vtable=
{
		.command_description="Creates the initial daemon state file for the authorization level daemon.",
		.command="create_daemon_state",
		.init=init_daemon_state_cmd_init,
		.parse_args=init_daemon_state_cmd_parse_args,
		.start=init_daemon_state_cmd_start,
		.deinit=init_daemon_state_cmd_deinit
};
